<?php

namespace App\Filament\Pages;

use App\Models\SocialLink;
use Filament\Actions\Action;
use Filament\Forms\Components\Actions\Action as ActionsAction;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Forms\Set;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class Socials extends Page
{
    protected static string $view = "filament.pages.socials";

    protected static ?string $model = SocialLink::class;

    protected static ?string $navigationIcon = "heroicon-o-share";

    public static function getNavigationGroup(): ?string
    {
        return __("navigation.socials");
    }

    public function getTitle(): string
    {
        return __("socials.socials");
    }

    public static function getNavigationLabel(): string
    {
        return __("socials.socials");
    }

    public function getActions(): array
    {
        return [
            Action::make(__("socials.save"))
                ->icon("heroicon-o-pencil-square")
                ->action(function () {
                    $this->validate();
                    $this->save();
                }),
        ];
    }

    public array $data = [];

    public function mount(): void
    {
        $platforms = [
            "whatsapp",
            "facebook",
            "tiktok",
            "twitter",
            "instagram",
            "linkedin",
            "youtube",
        ];

        $socials = [];
        foreach ($platforms as $platform) {
            $socials[$platform] = SocialLink::where(
                "platform",
                $platform
            )->first()?->link;
        }

        $this->form->fill(["socials" => $socials]);
    }

    public function form(Form $form): Form
    {
        $platforms = [
            "facebook" => "fab-facebook",
            "instagram" => "fab-instagram",
            "tiktok" => "fab-tiktok",
            "whatsapp" => "fab-whatsapp",
            "twitter" => "fab-twitter",
            "linkedin" => "fab-linkedin",
            "youtube" => "fab-youtube",
        ];

        $inputs = [];
        foreach ($platforms as $platform => $icon) {
            $inputs[] = TextInput::make("socials.$platform")
                ->label(__("socials." . $platform))
                ->placeholder(__("socials." . $platform . "-link"))
                ->prefixIcon($icon)
                ->url()
                ->suffixAction(
                    ActionsAction::make(__("socials.open"))
                        ->icon("heroicon-o-x-mark")
                        ->tooltip(__("socials.delete-link"))
                        ->action(function (Set $set, $state) use ($platform) {
                            $set("socials.$platform", null);
                        })
                );
        }

        return $form
            ->statePath("data")
            ->schema([
                Section::make(__("socials.manage"))
                    ->description(__("socials.description"))
                    ->aside()
                    ->schema($inputs),
            ]);
    }

    public function save(): void
    {
        try {
            DB::transaction(function () {
                foreach ($this->data["socials"] as $platform => $link) {
                    if (!$link) {
                        SocialLink::where("platform", $platform)->delete();
                        continue;
                    }

                    SocialLink::updateOrCreate(
                        ["platform" => Str::replace("\"", "", $platform)],
                        ["link" => $link]
                    );
                }
            });

            Notification::make("socials-updated")
                ->success()
                ->title(__("socials.success"))
                ->body(__("socials.socials-updated"))
                ->send();
        } catch (\Exception $e) {
            dd($e);
            Log::error($e->getMessage(), $e->getTrace());

            Notification::make("socials-error")
                ->danger()
                ->title(__("socials.error"))
                ->body(__("socials.social-error"))
                ->send();
        }
    }
}
