<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\Translatable\HasTranslations;

class News extends Model implements HasMedia
{
    use HasFactory, HasTranslations, InteractsWithMedia;

    protected $fillable = ["title", "body", "slug"];

    public $translatable = ["title", "body"];

    protected static function boot(): void
    {
        parent::boot();

        static::creating(function ($news) {
            $news->slug = static::generateUniqueSlug($news->title);
        });

        static::updating(function ($news) {
            if ($news->isDirty("title")) {
                $news->slug = static::generateUniqueSlug(
                    $news->title,
                    $news->id
                );
            }
        });
    }

    protected static function generateUniqueSlug(
        string $title,
        int $excludeId = null
    ): string {
        $slug = Str::slug($title);
        $originalSlug = $slug;
        $counter = 1;

        while (
            static::where("slug", $slug)
                ->when(
                    $excludeId,
                    fn($query) => $query->where("id", "!=", $excludeId)
                )
                ->exists()
        ) {
            $slug = "{$originalSlug}-{$counter}";
            $counter++;
        }

        return $slug;
    }

    public const MEDIA_COLLECTION_COVER = "cover";

    public function registerMediaCollections(): void
    {
        $fallback =
            "https://placehold.co/400x225.png?text=" .
            str_replace(" ", "%20", $this->title);

        $this->addMediaCollection(self::MEDIA_COLLECTION_COVER)->useFallbackUrl(
            $fallback
        );
    }
}
