<?php

namespace App\Services\AuthProviders;

use App\Contracts\AuthProvider;
use App\Enums\Role;
use App\Models\User;
use App\Exceptions\EmailAlreadyExistsException;
use Illuminate\Support\Facades\Auth;
use Laravel\Socialite\Facades\Socialite;

class GithubAuthProvider implements AuthProvider
{
    public function redirect(): string
    {
        return Socialite::driver('github')
            ->setScopes(['read:user', 'user:email'])
            ->stateless()
            ->redirect()
            ->getTargetUrl();
    }

    public function callback(): void
    {
        $githubUser = Socialite::driver('github')->user();

        if (User::where('github_id', '<>', $githubUser->id)->where('email', $githubUser->getEmail())->exists()) {
            throw new EmailAlreadyExistsException;
        }

        $user = User::updateOrCreate([
            'github_id' => $githubUser->id,
        ], [
            'name' => $githubUser->name,
            'email' => $githubUser->email,
            'github_token' => $githubUser->token,
            'github_refresh_token' => $githubUser->refreshToken,
        ]);

        $user->assignRole(Role::customer);
        $user->customer()->create();
        if ($githubUser->getAvatar()) {
            $user->addMediaFromUrl($githubUser->getAvatar())->toMediaCollection(User::MEDIA_COLLECTION_AVATAR);
        }

        Auth::login($user);
    }
}
