<?php

namespace App\Filament\Pages;

use App\Support\ChairmanMessagePage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ChairmanMessage extends Page
{
    protected static ?string $navigationIcon = "heroicon-o-document-text";

    protected static string $view = "filament.pages.chairman-message";

    public array $data = [];

    public static function getNavigationGroup(): ?string
    {
        return __("navigation.cms");
    }

    public function getTitle(): string|Htmlable
    {
        return __("chairman-message.chairman-message");
    }

    public static function getNavigationLabel(): string
    {
        return __("chairman-message.chairman-message");
    }

    public function mount(): void
    {
        $page = ChairmanMessagePage::make();

        $this->form->fill([
            "message" => $page->message()->getRaw("message"),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form->statePath("data")->schema([
            Forms\Components\Section::make(__("chairman-message.header"))
                ->description(__("chairman-message.header-description"))
                ->aside()
                ->schema([
                    Forms\Components\SpatieMediaLibraryFileUpload::make("cover")
                        ->label(__("chairman-message.cover-label"))
                        ->collection(
                            ChairmanMessagePage::MEDIA_COLLECTION_COVER
                        )
                        ->model(ChairmanMessagePage::make()->model())
                        ->required(),
                ]),
            Forms\Components\Section::make(__("chairman-message.message"))
                ->description(__("chairman-message.message-description"))
                ->aside()
                ->schema([
                    Forms\Components\SpatieMediaLibraryFileUpload::make(
                        "chairman"
                    )
                        ->label(__("chairman-message.chairman-label"))
                        ->collection(
                            ChairmanMessagePage::MEDIA_COLLECTION_CHAIRMAN
                        )
                        ->model(ChairmanMessagePage::make()->model())
                        ->required(),

                    Forms\Components\RichEditor::make("message")
                        ->label(__("chairman-message.message-label"))
                        ->placeholder(
                            __("chairman-message.message-placeholder")
                        )
                        ->required()
                        ->translatable(),
                ]),
        ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__("app.update"))
                ->icon("heroicon-o-arrow-path")
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        try {
            DB::beginTransaction();

            $page = ChairmanMessagePage::make();

            $page->setCover(collect($this->data["cover"])->first());

            $page->setChairman(collect($this->data["cover"])->first());

            $page->header()->patch([]);
            $page->message()->patch([
                "message" => $this->data["message"],
            ]);

            Notification::make()
                ->success()
                ->title(__("app.content-update-success"))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            Log::error("Failed to update Chairman Message data", [
                "message" => $e->getMessage(),
                "trace" => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__("app.content-update-failure"))
                ->send();
        }
    }
}
