<?php

namespace App\Filament\Pages;

use App\Support\ClientsPage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class Clients extends Page
{
    protected static ?string $navigationIcon = "heroicon-o-document-text";

    protected static string $view = "filament.pages.clients";

    public array $data = [];

    public static function getNavigationGroup(): ?string
    {
        return __("navigation.cms");
    }

    public function getTitle(): string|Htmlable
    {
        return __("clients.clients");
    }

    public static function getNavigationLabel(): string
    {
        return __("clients.clients");
    }

    public function mount(): void
    {
        $page = ClientsPage::make();

        $this->form->fill([
            "title" => $page->header()->getRaw("title"),
            "description" => $page->header()->getRaw("description"),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form->statePath("data")->schema([
            Forms\Components\Section::make(__("clients.header"))
                ->description(__("clients.header-description"))
                ->aside()
                ->schema([
                    Forms\Components\SpatieMediaLibraryFileUpload::make("cover")
                        ->label(__("clients.cover-label"))
                        ->collection(ClientsPage::MEDIA_COLLECTION_COVER)
                        ->model(ClientsPage::make()->model())
                        ->required(),

                    Forms\Components\TextInput::make("title")
                        ->label(__("clients.title-label"))
                        ->placeholder(__("clients.title-placeholder"))
                        ->required()
                        ->maxLength(255)
                        ->translatable(),

                    Forms\Components\TextInput::make("description")
                        ->label(__("clients.description-label"))
                        ->placeholder(__("clients.description-placeholder"))
                        ->required()
                        ->maxLength(255)
                        ->translatable(),
                ]),
        ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__("app.update"))
                ->icon("heroicon-o-arrow-path")
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        try {
            DB::beginTransaction();

            $page = ClientsPage::make();
            $page->setCover(collect($this->data["cover"])->first());

            $page->header()->patch([
                "title" => $this->data["title"],
                "description" => $this->data["description"],
            ]);

            Notification::make()
                ->success()
                ->title(__("app.content-update-success"))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            dd($e);
            Log::error("Failed to update Clients data", [
                "message" => $e->getMessage(),
                "trace" => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__("app.content-update-failure"))
                ->send();
        }
    }
}
