<?php

namespace App\Filament\Pages;

use App\Support\NewsPage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class News extends Page
{
    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    protected static string $view = 'filament.pages.news';

    public array $data = [];

    public static function getNavigationGroup(): ?string
    {
        return __('navigation.cms');
    }

    public function getTitle(): string|Htmlable
    {
        return __('news.news');
    }

    public static function getNavigationLabel(): string
    {
        return __('news.news');
    }

    public function mount(): void
    {
        $page = NewsPage::make();

        $this->form->fill([
            
            
            'title' => $page->header()->getRaw('title'),
            
            'description' => $page->header()->getRaw('description'),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->statePath('data')
            ->schema([
                Forms\Components\Section::make(__('news.header'))
                    ->description(__('news.header-description'))
                    ->aside()
                    ->schema([
                        
                        Forms\Components\SpatieMediaLibraryFileUpload::make('cover')
                            ->label(__('news.cover-label'))
                            ->collection(NewsPage::MEDIA_COLLECTION_COVER)
                            ->model(NewsPage::make()->model())
                            ->required()
                        
                        ,
                        
                        Forms\Components\TextInput::make('title')
                            ->label(__('news.title-label'))
                            ->placeholder(__('news.title-placeholder'))
                            ->required()
                            ->maxLength(255)
                            ->translatable()
                        ,
                        
                        Forms\Components\TextInput::make('description')
                            ->label(__('news.description-label'))
                            ->placeholder(__('news.description-placeholder'))
                            ->required()
                            ->maxLength(255)
                            ->translatable()
                        ,
                    ]),
            ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__('app.update'))
                ->icon('heroicon-o-arrow-path')
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        try {
            DB::beginTransaction();

            $page = NewsPage::make();
            $page->setCover(collect($this->data['cover'])->first());
            $page->header()->patch([
                'cover' => $this->data['cover'],
                'title' => $this->data['title'],
                'description' => $this->data['description'],
            ]);

            Notification::make()
                ->success()
                ->title(__('app.content-update-success'))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            Log::error('Failed to update News data', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__('app.content-update-failure'))
                ->send();
        }
    }
}
