<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\Translatable\HasTranslations;

class LocalTalent extends Model implements HasMedia
{
    use HasFactory, HasTranslations, InteractsWithMedia;

    protected $table = "local_talents";

    protected $fillable = ["name", "description", "slug"];

    public $translatable = ["name", "description"];

    public const MEDIA_COLLECTION_IMAGE = "image";

    protected static function boot(): void
    {
        parent::boot();

        static::creating(function ($localTalent) {
            if (empty($localTalent->slug)) {
                $localTalent->slug = static::generateUniqueSlug(
                    $localTalent->name
                );
            }
        });

        static::updating(function ($localTalent) {
            if ($localTalent->isDirty("name") && empty($localTalent->slug)) {
                $localTalent->slug = static::generateUniqueSlug(
                    $localTalent->name,
                    $localTalent->id
                );
            }
        });
    }

    protected static function generateUniqueSlug(
        string $name,
        int $excludeId = null
    ): string {
        $slug = Str::slug($name);
        $originalSlug = $slug;
        $counter = 1;

        while (
            static::where("slug", $slug)
                ->when(
                    $excludeId,
                    fn($query) => $query->where("id", "!=", $excludeId)
                )
                ->exists()
        ) {
            $slug = "{$originalSlug}-{$counter}";
            $counter++;
        }

        return $slug;
    }

    public function registerMediaCollections(): void
    {
        $fallback =
            "https://placehold.co/400x225.png?text=" .
            str_replace(" ", "%20", $this->name);

        $this->addMediaCollection(self::MEDIA_COLLECTION_IMAGE)->useFallbackUrl(
            $fallback
        );
    }
}
