<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\Translatable\HasTranslations;

class Project extends Model implements HasMedia
{
    use HasFactory, HasTranslations, InteractsWithMedia;

    protected $fillable = [
        "title",
        "subtitle",
        "number_of_staff",
        "number_of_locations",
        "starting_year",
        "contract_duration",
        "slug",
    ];

    public $translatable = ["title", "subtitle"];

    public const MEDIA_COLLECTION_LOGO = "logo";
    public const MEDIA_COLLECTION_IMAGE = "image";

    protected static function boot(): void
    {
        parent::boot();

        static::creating(function ($project) {
            $project->slug = static::generateUniqueSlug($project->title);
        });

        static::updating(function ($project) {
            if ($project->isDirty("title")) {
                $project->slug = static::generateUniqueSlug($project->title, $project->id);
            }
        });
    }

    protected static function generateUniqueSlug(string $title, int $excludeId = null): string
    {
        $slug = Str::slug($title);
        $originalSlug = $slug;
        $counter = 1;

        while (
            static::where("slug", $slug)
                ->when($excludeId, fn($query) => $query->where("id", "!=", $excludeId))
                ->exists()
        ) {
            $slug = "{$originalSlug}-{$counter}";
            $counter++;
        }

        return $slug;
    }

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection(self::MEDIA_COLLECTION_LOGO)
            ->singleFile()
            ->useFallbackUrl("https://ui-avatars.com/api/?name={$this->title}");

        $this->addMediaCollection(self::MEDIA_COLLECTION_IMAGE)
            ->singleFile()
            ->useFallbackUrl("https://placehold.co/400x225.png?text=" . str_replace(" ", "%20", $this->title));
    }
}
