<?php

namespace Tests\Feature\Support;

use App\Models\SitePage;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use App\Support\Page;
use App\Support\Section;

class HomePage extends Page
{
    public function key(): string
    {
        return 'home';
    }
}

class NonExistent extends Page
{
    public function key(): string
    {
        return 'non existent';
    }
};

class PageTest extends TestCase
{
    use RefreshDatabase;

    protected SitePage $page;
    protected HomePage $home;

    protected function setUp(): void
    {
        parent::setUp();

        $this->page = SitePage::factory()->create([
            'key' => 'home',
            'content' => $this->getPageContent(),
        ]);

        $this->home = new HomePage;
    }

    protected function getPageContent(): array
    {
        return [
            'hero' => [
                'title' => [
                    'en' => 'Hero Title',
                    'ar' => 'Arabic Hero Title',
                ],
                'description' => [
                    'en' => 'This is the homepage of the website',
                    'ar' => 'Arabic Description',
                ],
            ],
            'about' => [
                'title' => [
                    'en' => 'About us',
                    'ar' => 'Arabic About us',
                ],
                'description' => [
                    'en' => 'This is the about page of the website',
                    'ar' => 'Arabic Description',
                ],
            ],
            'seo' => [
                'title' => [
                    'en' => 'Homepage',
                    'ar' => 'Arabic Homepage',
                ],
                'description' => [
                    'en' => 'This is the homepage of the website',
                    'ar' => 'Arabic Description',
                ],
                'keywords' => ['homepage', 'website', 'home'],
            ],
        ];
    }

    public function test_it_returns_correct_home_key(): void
    {
        $this->assertEquals('home', $this->home->key());
    }

    public function test_it_checks_number_of_sections(): void
    {
        $this->assertCount(count($this->page->content), $this->home->sections());
    }

    public function test_it_returns_all_page_content(): void
    {
        $this->assertEquals($this->getExpectedContent(), $this->home->all());
    }

    public function test_it_retrieves_hero_section(): void
    {
        $hero = $this->home->get('hero');

        $this->assertEquals('hero', $hero->key());
        $this->assertEquals('home', $hero->parent()->key());
        $this->assertEquals('Hero Title', $hero->get('title'));
        $this->assertEquals([
            'title' => 'Hero Title',
            'description' => 'This is the homepage of the website',
        ], $hero->all());
    }

    public function test_it_throws_exception_for_missing_key(): void
    {
        $hero = $this->home->get('hero');
        $this->expectException(\Exception::class);
        $this->expectExceptionMessage('Key [foo] not found');
        $hero->get('foo');
    }

    public function test_it_returns_default_value_for_missing_key(): void
    {
        $hero = $this->home->get('hero');
        $this->assertEquals('default value', $hero->get('foo', 'default value'));
    }

    public function test_it_updates_hero_section(): void
    {
        $this->home->update('hero', ['title' => 'New Title']);
        $this->assertEquals('New Title', $this->home->get('hero')->get('title'));
        $this->assertEquals('This is the homepage of the website', $this->home->get('hero')->get('description'));
    }

    public function test_it_replaces_updated_values(): void
    {
        $this->home->update('hero', ['title' => 'New Title 2'], replace: true);
        $this->assertEquals('New Title 2', $this->home->get('hero')->get('title'));
        $this->assertEquals('', $this->home->get('hero')->get('description', ''));
    }

    public function test_it_patches_multiple_sections(): void
    {
        $this->home->patch([
            'hero' => ['title' => 'New Title 3', 'description' => 'New Description 3'],
            'about' => ['title' => 'New Title 4'],
        ]);

        $this->assertEquals('New Title 3', $this->home->get('hero')->get('title'));
        $this->assertEquals('New Description 3', $this->home->get('hero')->get('description'));
        $this->assertEquals('New Title 4', $this->home->get('about')->get('title'));
    }

    public function test_it_throws_exception_for_non_existing_section(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('Section [foo] not found');
        $this->home->get('foo');
    }

    public function test_it_doesnt_throw_exception_for_non_existing_section_when_patching(): void
    {
        $this->home->patch([
            'foo' => ['title' => 'New Title 3', 'description' => 'New Description 3'],
        ]);

        $this->assertEquals('New Title 3', $this->home->get('foo')->get('title'));
        $this->assertEquals('New Description 3', $this->home->get('foo')->get('description'));
    }

    public function test_it_doesnt_throw_an_exception_if_a_page_with_given_key_doesnt_exist_in_database(): void
    {
        $result = NonExistent::make()->get('bar', ['baz' => 'default bar']);
        $this->assertEquals(
            new Section('bar', ['baz' => 'default bar'], NonExistent::make()),
            $result
        );
    }

    private function getExpectedContent(): array
    {
        return [
            'hero' => [
                'title' => 'Hero Title',
                'description' => 'This is the homepage of the website',
            ],
            'about' => [
                'title' => 'About us',
                'description' => 'This is the about page of the website',
            ],
            'seo' => [
                'title' => 'Homepage',
                'description' => 'This is the homepage of the website',
                'keywords' => ['homepage', 'website', 'home'],
            ],
        ];
    }
}
